// background.js
var SUPABASE_URL = "https://prgwbpnnpfbnlbzdlcdc.supabase.co";
var SUPABASE_ANON_KEY = "sb_publishable_nh-tk7YqZ2JMKZO8Ufk8ew_F0Fe7GEl";
var SUPABASE_SESSION_KEY = "supabaseSession";
async function getStoredSession() {
  const result = await chrome.storage.local.get(SUPABASE_SESSION_KEY);
  return result[SUPABASE_SESSION_KEY] ?? null;
}
async function storeSession(session) {
  await chrome.storage.local.set({ [SUPABASE_SESSION_KEY]: session });
}
async function signInAnonymously() {
  const res = await fetch(`${SUPABASE_URL}/auth/v1/token?grant_type=anonymous`, {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
      "apikey": SUPABASE_ANON_KEY
    },
    body: JSON.stringify({})
  });
  if (!res.ok) throw new Error(`Anonymous sign-in failed: ${res.status}`);
  const data = await res.json();
  await storeSession(data);
  return data;
}
async function refreshSession(refreshToken) {
  const res = await fetch(`${SUPABASE_URL}/auth/v1/token?grant_type=refresh_token`, {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
      "apikey": SUPABASE_ANON_KEY
    },
    body: JSON.stringify({ refresh_token: refreshToken })
  });
  if (!res.ok) return null;
  const data = await res.json();
  await storeSession(data);
  return data;
}
async function getOrCreateSession() {
  const session = await getStoredSession();
  if (!session) return signInAnonymously();
  const nowSecs = Math.floor(Date.now() / 1e3);
  if ((session.expires_at ?? 0) - nowSecs < 60) {
    return await refreshSession(session.refresh_token).catch(() => null) ?? signInAnonymously();
  }
  return session;
}
async function trackVibeCheck() {
  try {
    const session = await getOrCreateSession();
    if (!session?.access_token) return;
    await fetch(`${SUPABASE_URL}/rest/v1/rpc/increment_vibecheck_count`, {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        "apikey": SUPABASE_ANON_KEY,
        "Authorization": `Bearer ${session.access_token}`
      },
      body: JSON.stringify({ p_user_id: session.user.id })
    });
  } catch {
  }
}
chrome.action.onClicked.addListener((tab) => {
  chrome.sidePanel.open({ windowId: tab.windowId });
});
function setPendingPost(post) {
  return chrome.storage.session.set({ pendingPost: post });
}
function consumePendingPost() {
  return chrome.storage.session.get("pendingPost").then(({ pendingPost: post }) => {
    chrome.storage.session.remove("pendingPost");
    return post ?? null;
  });
}
function setPendingHistoryUrn(urn) {
  return chrome.storage.session.set({ pendingHistoryUrn: urn });
}
function consumePendingHistoryUrn() {
  return chrome.storage.session.get("pendingHistoryUrn").then(({ pendingHistoryUrn: urn }) => {
    chrome.storage.session.remove("pendingHistoryUrn");
    return urn ?? null;
  });
}
chrome.runtime.onConnect.addListener((port) => {
  port.onMessage.addListener(() => {
  });
});
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.type === "SHOW_HISTORY_ENTRY") {
    if (sender?.tab?.windowId) {
      chrome.sidePanel.open({ windowId: sender.tab.windowId }).catch(() => {
      });
    }
    setPendingHistoryUrn(message.urn).then(() => {
      chrome.runtime.sendMessage({ type: "DIRECT_HISTORY_ENTRY", urn: message.urn }).catch(() => {
      });
      sendResponse({ ok: true });
    });
    return true;
  }
  if (message.type === "GET_PENDING_HISTORY") {
    consumePendingHistoryUrn().then((urn) => sendResponse({ urn }));
    return true;
  }
  if (message.type === "BUTTON_UPDATE") {
    chrome.tabs.query({ active: true, currentWindow: true }).then(([tab]) => {
      if (tab) {
        chrome.tabs.sendMessage(tab.id, { type: "UPDATE_POST_BUTTON", urn: message.urn, verdict: message.verdict }).catch(() => {
        });
      }
    });
    sendResponse({ ok: true });
    return true;
  }
  if (message.type === "ANALYSE_FROM_PAGE") {
    if (sender?.tab?.windowId) {
      chrome.sidePanel.open({ windowId: sender.tab.windowId }).catch(() => {
      });
    }
    setPendingPost({ text: message.text, meta: message.meta }).then(() => {
      chrome.runtime.sendMessage({ type: "DIRECT_ANALYSE", text: message.text, meta: message.meta }).catch(() => {
      });
      sendResponse({ ok: true });
    });
    return true;
  }
  if (message.type === "GET_PENDING_POST") {
    consumePendingPost().then((post) => sendResponse({ post }));
    return true;
  }
  if (message.type === "GRAB_POST") {
    chrome.tabs.query({ active: true, currentWindow: true }).then(async ([tab]) => {
      if (!tab) {
        sendResponse({ found: false, reason: "no_tab" });
        return;
      }
      try {
        const result = await chrome.tabs.sendMessage(tab.id, { type: "GRAB_POST" });
        sendResponse(result || { found: false, reason: "no_result" });
      } catch {
        sendResponse({ found: false, reason: "no_content_script" });
      }
    });
    return true;
  }
  if (message.type === "ANALYSE_POST") {
    handleAnalysePost(message.postText, message.postMeta).then((result) => sendResponse(result)).catch((err) => sendResponse({ type: "ERROR", message: String(err) }));
    return true;
  }
});
async function handleAnalysePost(postText, postMeta) {
  const result = await callGrokApi(postText, postMeta);
  if (result.type === "ANALYSIS_RESULT") {
    trackVibeCheck();
  }
  return result;
}
async function callGrokApi(postText, postMeta) {
  let res;
  try {
    res = await fetch(`${SUPABASE_URL}/functions/v1/analyse-post`, {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        "apikey": SUPABASE_ANON_KEY
      },
      body: JSON.stringify({ postText, postMeta })
    });
  } catch (err) {
    return { type: "API_ERROR", code: 0, detail: err.message };
  }
  if (!res.ok) {
    const code = res.status;
    let detail = res.statusText;
    try {
      detail = (await res.json()).error || detail;
    } catch {
    }
    return { type: "API_ERROR", code, detail };
  }
  let result;
  try {
    result = await res.json();
  } catch {
    return { type: "PARSE_ERROR", raw: "" };
  }
  return result;
}
