// LinkedIn post container selectors, tried in order (most specific → broadest fallback)
const POST_CONTAINERS = [
  '.feed-shared-update-v2',
  '.occludable-update',
  '[data-finite-scroll-hotspot]',
  'article',
];

// Post text selectors, tried in order of reliability
const TEXT_SELECTORS = [
  '[data-test-id="main-feed-activity-card__commentary"]',
  '.feed-shared-text',
  '.feed-shared-update-v2__description',
  '.attributed-text-segment-list__content',
  '.break-words',
];

// Author name selectors — try the inner aria-hidden span first (LinkedIn nests
// the visible text there; innerText on the parent element returns empty)
const AUTHOR_NAME_SELECTORS = [
  '.update-components-actor__name span[aria-hidden="true"]',
  '.update-components-actor__name',
  '.update-components-actor__title span[aria-hidden="true"]',
  '.update-components-actor__title',
  'a[data-field="actor_name"] span[aria-hidden="true"]',
  'a[data-field="actor_name"]',
  '.feed-shared-actor__name span[aria-hidden="true"]',
  '.feed-shared-actor__name',
];

// Author profile link selectors — used to extract the vanity ID from the href
const AUTHOR_LINK_SELECTORS = [
  'a[data-field="actor_name"]',
  '.update-components-actor__meta-link',
  '.feed-shared-actor__container-link',
];

// Author headline/title selectors (job title line beneath name)
const AUTHOR_HEADLINE_SELECTORS = [
  '.update-components-actor__description span[aria-hidden="true"]',
  '.update-components-actor__description',
  '.feed-shared-actor__description',
];

// Relative timestamp selectors — target the short form span first ("2d", "1w")
// to avoid the long "• 1 week ago • Visible to anyone on or off LinkedIn" string
const TIMESTAMP_SELECTORS = [
  '.update-components-actor__sub-description span[aria-hidden="true"]',
  '.update-components-actor__sub-description',
  '.feed-shared-actor__sub-description span[aria-hidden="true"]',
  '.feed-shared-actor__sub-description',
];

// Post action bar selectors (Like / Comment / Repost / Send row)
const ACTION_BAR_SELECTORS = [
  '.feed-shared-social-action-bar',
  '.update-v2-social-activity',
  '.feed-shared-social-actions',
];

const BTN_SVG = `<svg class="lo-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true"><polygon points="13 2 3 14 12 14 11 22 21 10 12 10 13 2"/></svg>`;

const VERDICT_DISPLAY = {
  GREEN:  { bg: '#22c55e', emoji: '👌' },
  YELLOW: { bg: '#f59e0b', emoji: '🤔' },
  RED:    { bg: '#7f1d1d', emoji: '☠️' },
  SLOP:   { bg: '#7c3aed', emoji: '🤖' },
};

function extractText(container) {
  for (const sel of TEXT_SELECTORS) {
    const el = container.querySelector(sel);
    if (el) {
      const text = el.innerText.trim();
      if (text.length >= 20) return text;
    }
  }
  // Fallback: LinkedIn wraps post body text in dir="ltr" spans; pick the
  // longest one (to avoid grabbing short UI strings like author names).
  let best = "";
  for (const span of container.querySelectorAll('span[dir="ltr"]')) {
    const text = span.innerText.trim();
    if (text.length > best.length) best = text;
  }
  if (best.length >= 20) return best;
  return null;
}

function firstInnerText(container, selectors) {
  for (const sel of selectors) {
    const el = container.querySelector(sel);
    if (el) {
      const text = el.innerText.trim();
      if (text) return text;
    }
  }
  return null;
}

// Extract the LinkedIn URN from the post container's data attribute.
// data-urn="urn:li:activity:7296401234567890"
function extractUrn(container) {
  let el = container;
  while (el) {
    if (el.dataset && el.dataset.urn) return el.dataset.urn;
    el = el.parentElement;
  }
  return null;
}

function urnToUrl(urn) {
  if (!urn) return null;
  return `https://www.linkedin.com/feed/update/${urn}`;
}

function extractAuthorProfileUrl(container) {
  for (const sel of AUTHOR_LINK_SELECTORS) {
    const el = container.querySelector(sel);
    if (el && el.href) return el.href;
  }
  return null;
}

function profileUrlToId(url) {
  if (!url) return null;
  const match = url.match(/linkedin\.com\/in\/([^/?#]+)/);
  return match ? match[1] : null;
}

// Converts LinkedIn's relative timestamp text ("2d", "1w", "3mo") to an
// approximate ISO date string based on the current time.
function relativeToISO(text) {
  const match = text.trim().match(/^(\d+)\s*(m|h|d|w|mo|yr)\.?$/i);
  if (!match) return null;
  const n = parseInt(match[1], 10);
  const unit = match[2].toLowerCase();
  const msMap = { m: 6e4, h: 36e5, d: 864e5, w: 6048e5, mo: 2592e6, yr: 31536e6 };
  const ms = msMap[unit];
  if (!ms) return null;
  return new Date(Date.now() - n * ms).toISOString();
}

// Returns an ISO timestamp for the post. Prefers the real datetime attribute
// from a <time> element; falls back to converting LinkedIn's relative text.
function extractTimestamp(container) {
  const timeEl = container.querySelector('time[datetime]');
  if (timeEl) {
    const dt = timeEl.getAttribute('datetime');
    if (dt) {
      try {
        const d = new Date(dt);
        if (!isNaN(d.getTime())) return d.toISOString();
      } catch {}
    }
  }
  const rel = firstInnerText(container, TIMESTAMP_SELECTORS);
  if (!rel) return null;
  return relativeToISO(rel) || rel;
}

function extractPostMeta(container) {
  const urn = extractUrn(container);
  const profileUrl = extractAuthorProfileUrl(container);
  return {
    author:    firstInnerText(container, AUTHOR_NAME_SELECTORS) ?? 'Unknown',
    authorId:  profileUrlToId(profileUrl),
    headline:  firstInnerText(container, AUTHOR_HEADLINE_SELECTORS),
    timestamp: extractTimestamp(container),
    urn,
    postUrl:   urnToUrl(urn),
  };
}

// Returns all post containers using the most specific selector that yields results
function getAllPostContainers() {
  for (const sel of POST_CONTAINERS) {
    const els = document.querySelectorAll(sel);
    if (els.length > 0) return Array.from(els);
  }
  return [];
}

// After an analysis completes the side panel notifies us so we can immediately
// update any visible button for that post to its verdict colour.
chrome.runtime.onMessage.addListener((msg) => {
  if (msg.type !== 'UPDATE_POST_BUTTON') return;
  const { urn, verdict } = msg;
  getAllPostContainers().forEach((container) => {
    if (extractUrn(container) !== urn) return;
    const btn = container.querySelector('[data-vibecheck="true"]');
    if (btn && !btn.dataset.verdict) applyVerdictToButton(btn, verdict, urn);
  });
});

// ── Button injection into LinkedIn's action bar ───────────────────────────────

function injectStyles() {
  if (document.getElementById('vibecheck-styles')) return;
  const style = document.createElement('style');
  style.id = 'vibecheck-styles';
  style.textContent = `
    /* Icon-only action bar button — matches native button sizing */
    .vibecheck-analyse-btn {
      display: inline-flex !important;
      align-items: center !important;
      justify-content: center !important;
      padding: 6px 8px !important;
      max-height: 48px !important;
      flex: 1 !important;
      border: none !important;
      border-radius: 4px !important;
      background: transparent !important;
      color: #7c3aed !important;
      cursor: pointer !important;
      transition: background 0.15s, color 0.15s !important;
    }
    .vibecheck-analyse-btn .lo-icon {
      display: inline-block !important;
      width: 20px !important;
      height: 20px !important;
      flex-shrink: 0 !important;
    }
    .vibecheck-analyse-btn:hover {
      background: rgba(124,58,237,0.08) !important;
      color: #6d28d9 !important;
    }
    .vibecheck-analyse-btn:active,
    .vibecheck-analyse-btn.vibecheck-loading {
      background: rgba(124,58,237,0.14) !important;
      color: #5b21b6 !important;
    }
    /* Verdict state — coloured background with emoji */
    .vibecheck-analyse-btn[data-verdict] {
      font-size: 15px !important;
      line-height: 1 !important;
      border-radius: 6px !important;
    }
    .vibecheck-analyse-btn[data-verdict="GREEN"]  { background: #22c55e !important; }
    .vibecheck-analyse-btn[data-verdict="YELLOW"] { background: #f59e0b !important; }
    .vibecheck-analyse-btn[data-verdict="RED"]    { background: #7f1d1d !important; }
    .vibecheck-analyse-btn[data-verdict="SLOP"]   { background: #7c3aed !important; }
    .vibecheck-analyse-btn[data-verdict="GREEN"]:hover  { background: #16a34a !important; }
    .vibecheck-analyse-btn[data-verdict="YELLOW"]:hover { background: #d97706 !important; }
    .vibecheck-analyse-btn[data-verdict="RED"]:hover    { background: #5b1111 !important; }
    .vibecheck-analyse-btn[data-verdict="SLOP"]:hover   { background: #6d28d9 !important; }
  `;
  document.head.appendChild(style);
}

function findActionBar(postContainer) {
  for (const sel of ACTION_BAR_SELECTORS) {
    const el = postContainer.querySelector(sel);
    if (el) return el;
  }
  // Fallback: find the Like button and walk up to its action bar wrapper
  const buttons = postContainer.querySelectorAll('button');
  for (const btn of buttons) {
    const label = (btn.getAttribute('aria-label') || btn.textContent || '').toLowerCase();
    if (label.includes('like') && !label.includes('unlike')) {
      return btn.closest('[class*="action"]') || btn.parentElement?.parentElement || null;
    }
  }
  return null;
}

async function findHistoryEntry(urn) {
  if (!urn) return null;
  try {
    const { postHistory = [] } = await chrome.storage.local.get('postHistory');
    return postHistory.find(e => e.meta?.urn === urn) ?? null;
  } catch {
    return null;
  }
}

// Replaces a button with a verdict-coloured emoji version (clones without children
// to strip the SVG, then sets the emoji and new click handler).
function applyVerdictToButton(btn, verdict, urn) {
  const { emoji } = VERDICT_DISPLAY[verdict] || VERDICT_DISPLAY.YELLOW;
  const newBtn = btn.cloneNode(false); // attributes only — no SVG child
  newBtn.dataset.verdict = verdict;
  newBtn.innerHTML = emoji;
  newBtn.setAttribute('aria-label', 'View previous analysis');
  newBtn.addEventListener('click', (e) => {
    e.stopPropagation();
    if (!chrome.runtime?.id) return;
    try {
      chrome.runtime.sendMessage({ type: 'SHOW_HISTORY_ENTRY', urn }).catch(() => {});
    } catch {}
  });
  btn.parentElement?.replaceChild(newBtn, btn);
  return newBtn;
}

async function injectAnalyseButton(postContainer) {
  const urn = extractUrn(postContainer);
  const existingBtn = postContainer.querySelector('[data-vibecheck="true"]');

  if (existingBtn) {
    // Already in final (verdict) state — nothing to do
    if (existingBtn.dataset.verdict) return;
    // Was injected before this post was analysed — check if history has caught up
    const entry = await findHistoryEntry(urn);
    if (entry) applyVerdictToButton(existingBtn, entry.data.verdict, urn);
    return;
  }

  const actionBar = findActionBar(postContainer);
  if (!actionBar) return;

  const btn = document.createElement('button');
  btn.setAttribute('data-vibecheck', 'true');
  btn.className = 'vibecheck-analyse-btn';
  btn.innerHTML = BTN_SVG;

  const entry = await findHistoryEntry(urn);
  if (entry) {
    const { emoji } = VERDICT_DISPLAY[entry.data.verdict] || VERDICT_DISPLAY.YELLOW;
    btn.dataset.verdict = entry.data.verdict;
    btn.innerHTML = emoji;
    btn.setAttribute('aria-label', 'View previous analysis');
    btn.addEventListener('click', (e) => {
      e.stopPropagation();
      if (!chrome.runtime?.id) return;
      try {
        chrome.runtime.sendMessage({ type: 'SHOW_HISTORY_ENTRY', urn }).catch(() => {});
      } catch {}
    });
  } else {
    btn.setAttribute('aria-label', 'Analyse post with VibeCheck');
    btn.addEventListener('click', (e) => {
      e.stopPropagation();
      if (!chrome.runtime?.id) return;
      const text = extractText(postContainer);
      if (!text) return;
      btn.classList.add('vibecheck-loading');
      setTimeout(() => btn.classList.remove('vibecheck-loading'), 1800);
      try {
        chrome.runtime
          .sendMessage({ type: 'ANALYSE_FROM_PAGE', text, meta: extractPostMeta(postContainer) })
          .catch(() => {});
      } catch {}
    });
  }

  // Append directly into the action bar so it remains in its original DOM
  // context. Wrapping the action bar in a new element disrupts LinkedIn's
  // layout — causing native buttons to shift from horizontal to vertical.
  actionBar.appendChild(btn);
}

function processNewPosts() {
  getAllPostContainers().forEach((el) => {
    injectAnalyseButton(el);
  });
}

// Throttled MutationObserver to catch dynamically added posts from infinite scroll
let injectThrottleTimer = null;
const feedMutationObserver = new MutationObserver(() => {
  clearTimeout(injectThrottleTimer);
  injectThrottleTimer = setTimeout(processNewPosts, 800);
});
feedMutationObserver.observe(document.body, { childList: true, subtree: true });

function removeListeners() {
  clearInterval(keepalivePingInterval);
  keepalivePort = null;
  feedMutationObserver.disconnect();
  clearInterval(rescanInterval);
  clearTimeout(injectThrottleTimer);
}

// Initial scan + periodic rescan to catch posts added by infinite scroll
injectStyles();
processNewPosts();
const rescanInterval = setInterval(processNewPosts, 3000);

// Keep the background service worker alive with a periodic ping.
// MV3 SWs go idle after ~30s of inactivity, which causes two problems:
//   1. The port disconnects, killing observers (button injection stops)
//   2. The user gesture token expires before sidePanel.open() is called (clicks do nothing)
// Pinging every 20s keeps the SW warm so both issues are avoided.
//
// If the extension context is genuinely invalidated (reload/update),
// chrome.runtime.id will be undefined and we clean up instead.
let keepalivePort = null;
let keepalivePingInterval = null;

function connectKeepalive() {
  if (!chrome.runtime?.id) {
    removeListeners();
    return;
  }
  try {
    keepalivePort = chrome.runtime.connect({ name: 'vibecheck-content' });
    keepalivePingInterval = setInterval(() => {
      try {
        keepalivePort?.postMessage({ type: 'ping' });
      } catch {
        clearInterval(keepalivePingInterval);
      }
    }, 20000);
    keepalivePort.onDisconnect.addListener(() => {
      clearInterval(keepalivePingInterval);
      keepalivePort = null;
      if (chrome.runtime?.id) {
        // SW went idle despite the ping — reconnect immediately
        connectKeepalive();
      } else {
        // Context genuinely invalidated — stop all activity
        removeListeners();
      }
    });
  } catch {
    removeListeners();
  }
}

connectKeepalive();
