// ── Supabase config ───────────────────────────────────────────────────────────
const SUPABASE_URL      = "https://prgwbpnnpfbnlbzdlcdc.supabase.co";
const SUPABASE_ANON_KEY = "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJzdXBhYmFzZSIsInJlZiI6InByZ3dicG5ucGZibmxiemRsY2RjIiwicm9sZSI6ImFub24iLCJpYXQiOjE3NzE4Nzc4OTIsImV4cCI6MjA4NzQ1Mzg5Mn0.ukboEepWsgvjkO_IoGYgJ5BT09IbEEg-dvQ61FMm5EM";

async function registerActivationInSupabase(email) {
  try {
    await fetch(`${SUPABASE_URL}/rest/v1/activated_users`, {
      method: "POST",
      headers: {
        "apikey": SUPABASE_ANON_KEY,
        "Authorization": `Bearer ${SUPABASE_ANON_KEY}`,
        "Content-Type": "application/json",
        "Prefer": "return=minimal",
      },
      body: JSON.stringify({ email }),
    });
    // 409 on duplicate email is fine — user already registered on another device.
  } catch (_) {
    // Network failure must never block local activation.
  }
}

const stateActivation = document.getElementById("stateActivation");
const activationEmail = document.getElementById("activationEmail");
const activationError = document.getElementById("activationError");
const activateBtn     = document.getElementById("activateBtn");
const stateReady    = document.getElementById("stateReady");
const stateLoading  = document.getElementById("stateLoading");
const stateResults  = document.getElementById("stateResults");
const stateError    = document.getElementById("stateError");
const stateHistory  = document.getElementById("stateHistory");
const historyToggle = document.getElementById("historyToggle");

let activated = false;
const historyList   = document.getElementById("historyList");
const historyEmpty  = document.getElementById("historyEmpty");
const clearHistoryBtn = document.getElementById("clearHistoryBtn");
const verdictBanner = document.getElementById("verdictBanner");
const postAttribution = document.getElementById("postAttribution");
const scoreGrid     = document.getElementById("scoreGrid");
const summaryBlock  = document.getElementById("summaryBlock");
const flagsSection  = document.getElementById("flagsSection");
const flagsList     = document.getElementById("flagsList");
const errorMessage  = document.getElementById("errorMessage");
const retryBtn      = document.getElementById("retryBtn");

const VERDICT_CONFIG = {
  GREEN:  { bg: "#22c55e", label: "👌 Giving Trust Vibes" },
  YELLOW: { bg: "#f59e0b", label: "🤔 Healthy Scepticism" },
  RED:    { bg: "#7f1d1d", label: "☠️ Doom Scroll Alert" },
  SLOP:   { bg: "#7c3aed", label: "🤖 Straight Up Slop" },
};

const SCORE_META = {
  receipt_check: { label: "Receipt Check", inverted: false },
  real_world_iq: { label: "Real-World IQ",  inverted: false },
  bs_score:      { label: "BS Score",        inverted: true  },
  clout_farming: { label: "Clout Farming",   inverted: true  },
  slop_score:    { label: "Slop Score",      inverted: true  },
};

// ── State management ─────────────────────────────────────────────────────────

function showActivationState() {
  stateActivation.classList.remove("hidden");
  stateReady.classList.add("hidden");
  stateLoading.classList.add("hidden");
  stateResults.classList.add("hidden");
  stateError.classList.add("hidden");
  stateHistory.classList.add("hidden");
  historyToggle.classList.remove("active");
}

function showState(ready, loading, results, err, history = false) {
  stateActivation.classList.add("hidden");
  stateReady.classList.toggle("hidden", !ready);
  stateLoading.classList.toggle("hidden", !loading);
  stateResults.classList.toggle("hidden", !results);
  stateError.classList.toggle("hidden", !err);
  stateHistory.classList.toggle("hidden", !history);
  historyToggle.classList.toggle("active", history);
}

function showError(msg, canRetry = false) {
  errorMessage.textContent = msg;
  retryBtn.classList.toggle("hidden", !canRetry);
  showState(false, false, false, true);
}

function setLoading(loading) {
  showState(!loading, loading, false, false);
}

// ── Results rendering ─────────────────────────────────────────────────────────

function renderResults(data) {
  const cfg = VERDICT_CONFIG[data.verdict] || VERDICT_CONFIG.YELLOW;
  verdictBanner.style.backgroundColor = cfg.bg;
  verdictBanner.textContent = cfg.label;
  verdictBanner.className = "verdict-banner";

  scoreGrid.innerHTML = "";

  const dimensions = ["receipt_check", "real_world_iq", "bs_score", "clout_farming", "slop_score"];
  dimensions.forEach((key) => {
    const { label, inverted } = SCORE_META[key];
    const score = data[key];
    const colorKey = inverted
      ? (score >= 7 ? "red" : score >= 4 ? "amber" : "green")
      : (score <= 3 ? "red" : score <= 6 ? "amber" : "green");
    const card = document.createElement("div");
    card.className = `score-card tint-${colorKey}`;
    card.innerHTML = `
      <span class="score-label">${label}</span>
      <span class="score-value val-${colorKey}">${score}</span>
      <span class="score-bar-track"><span class="score-bar bar-${colorKey}" style="width: ${(score / 10) * 100}%"></span></span>
    `;
    scoreGrid.appendChild(card);
  });

  summaryBlock.innerHTML = `<p class="summary-text">${escapeHtml(data.summary || "")}</p>`;

  const flags = data.flags && Array.isArray(data.flags) ? data.flags.slice(0, 3) : [];
  if (flags.length > 0) {
    flagsSection.classList.remove("hidden");
    flagsList.innerHTML = flags.map(f => `<span class="flag-pill">🚩 ${escapeHtml(f)}</span>`).join("");
  } else {
    flagsSection.classList.add("hidden");
  }

  renderPostAttribution(capturedMeta);
  showState(false, false, true, false);
}

function formatDisplayDate(ts) {
  if (!ts) return null;
  const d = new Date(ts);
  if (isNaN(d.getTime())) return ts;
  return d.toLocaleDateString("en-GB", { day: "numeric", month: "short", year: "numeric" });
}

function renderPostAttribution(meta) {
  const author = meta.author && meta.author !== "Unknown" ? meta.author : null;
  const headline = meta.headline || null;
  const timestamp = meta.timestamp ? formatDisplayDate(meta.timestamp) : null;
  const postUrl = meta.postUrl || null;

  if (!author && !timestamp) {
    postAttribution.classList.add("hidden");
    return;
  }

  const authorHtml = author
    ? (postUrl
        ? `<a class="pa-author" href="${escapeHtml(postUrl)}" target="_blank" rel="noopener">${escapeHtml(author)}</a>`
        : `<span class="pa-author">${escapeHtml(author)}</span>`)
    : "";
  const headlineHtml = headline ? `<span class="pa-headline">${escapeHtml(headline)}</span>` : "";
  const timeHtml = timestamp ? `<span class="pa-time">${escapeHtml(timestamp)}</span>` : "";

  postAttribution.innerHTML = `
    <div class="pa-left">
      ${authorHtml}
      ${headlineHtml}
    </div>
    ${timeHtml}
  `;
  postAttribution.classList.remove("hidden");
}


function escapeHtml(s) {
  const div = document.createElement("div");
  div.textContent = s;
  return div.innerHTML;
}

// ── History ───────────────────────────────────────────────────────────────────

const MAX_HISTORY = 30;

function timeAgo(isoString) {
  const mins = Math.floor((Date.now() - new Date(isoString).getTime()) / 60000);
  if (mins < 1)  return "just now";
  if (mins < 60) return `${mins}m ago`;
  const hrs = Math.floor(mins / 60);
  if (hrs < 24)  return `${hrs}h ago`;
  const days = Math.floor(hrs / 24);
  if (days < 7)  return `${days}d ago`;
  return `${Math.floor(days / 7)}w ago`;
}

async function saveToHistory(postText, meta, data) {
  const { postHistory = [] } = await chrome.storage.local.get("postHistory");
  const entry = { id: Date.now(), savedAt: new Date().toISOString(), meta: { ...meta }, postText, data: { ...data } };
  await chrome.storage.local.set({ postHistory: [entry, ...postHistory].slice(0, MAX_HISTORY) });
}

function renderHistoryList(items) {
  if (!items || items.length === 0) {
    historyList.innerHTML = "";
    historyEmpty.classList.remove("hidden");
    return;
  }
  historyEmpty.classList.add("hidden");
  historyList.innerHTML = items.map((item, i) => `
    <div class="history-item history-item--${item.data?.verdict || "YELLOW"}" data-idx="${i}">
      <div class="history-item-author">
        <span>${escapeHtml(item.meta?.author || "Unknown")}</span>
        <span class="history-item-time">${timeAgo(item.savedAt)}</span>
      </div>
      <div class="history-item-summary">${escapeHtml(item.data?.summary || "")}</div>
    </div>
  `).join("");
  historyList.querySelectorAll(".history-item").forEach((el) => {
    el.addEventListener("click", () => {
      const item = items[parseInt(el.dataset.idx, 10)];
      capturedMeta = item.meta || { ...EMPTY_META };
      capturedPostText = item.postText || "";
      renderResults(item.data);
    });
  });
}

historyToggle.addEventListener("click", async () => {
  if (!activated) return;
  if (!stateHistory.classList.contains("hidden")) {
    showState(true, false, false, false, false);
    return;
  }
  const { postHistory = [] } = await chrome.storage.local.get("postHistory");
  renderHistoryList(postHistory);
  showState(false, false, false, false, true);
});

clearHistoryBtn.addEventListener("click", async () => {
  await chrome.storage.local.set({ postHistory: [] });
  renderHistoryList([]);
});

// ── Captured post state ───────────────────────────────────────────────────────

const EMPTY_META = { author: "Unknown", authorId: null, headline: null, timestamp: null, urn: null, postUrl: null };
let capturedMeta = { ...EMPTY_META };
let capturedPostText = "";

// ── Message listener (injected LinkedIn button → side panel) ──────────────────

chrome.runtime.onMessage.addListener((msg) => {
  if (msg.type === "DIRECT_HISTORY_ENTRY") {
    (async () => {
      const { postHistory = [] } = await chrome.storage.local.get("postHistory");
      const entry = postHistory.find(e => e.meta?.urn === msg.urn);
      if (!entry) return;
      capturedMeta = entry.meta || { ...EMPTY_META };
      capturedPostText = entry.postText || "";
      renderResults(entry.data);
    })();
    return;
  }

  if (msg.type === "DIRECT_ANALYSE") {
    capturedMeta = msg.meta || { ...EMPTY_META };
    capturedPostText = msg.text || "";
    runAnalysis();
    return;
  }
});

// ── Retry button ──────────────────────────────────────────────────────────────

retryBtn.addEventListener("click", runAnalysis);

// ── Core vibe check ───────────────────────────────────────────────────────────

async function runAnalysis() {
  const postText = capturedPostText.trim();
  if (!postText || postText.length < 10) {
    showError("Tap ⚡ on a LinkedIn post to run a vibe check.", false);
    return;
  }
  setLoading(true);
  const result = await chrome.runtime.sendMessage({
    type: "ANALYSE_POST",
    postText,
    postMeta: capturedMeta,
  });
  if (result.type === "API_ERROR") {
    const detail = result.detail ? ` (${result.detail})` : "";
    if (result.code === 429) showError("Rate limit hit — wait a moment and try again.", true);
    else if (result.code >= 500) showError(`Grok's having a moment. Try again shortly.${detail}`, true);
    else showError(`Something went wrong. Try again.${detail}`, true);
    return;
  }
  if (result.type === "PARSE_ERROR") {
    showError("Couldn't read the vibe check result. Try again.", true);
    return;
  }
  if (result.type === "ANALYSIS_RESULT" && result.data) {
    await saveToHistory(postText, capturedMeta, result.data);
    if (capturedMeta.urn) {
      chrome.runtime.sendMessage({
        type: "BUTTON_UPDATE",
        urn: capturedMeta.urn,
        verdict: result.data.verdict,
      }).catch(() => {});
    }
    renderResults(result.data);
    return;
  }
  showError("Something went wrong. Try again.", true);
}

// ── Startup: consume any pending post/history from session storage ────────────

async function checkPendingPost() {
  const result = await chrome.runtime.sendMessage({ type: "GET_PENDING_POST" });
  if (result?.post) {
    capturedMeta = result.post.meta || { ...EMPTY_META };
    capturedPostText = result.post.text || "";
    runAnalysis();
  }
}

async function checkPendingHistory() {
  const result = await chrome.runtime.sendMessage({ type: "GET_PENDING_HISTORY" });
  if (result?.urn) {
    const { postHistory = [] } = await chrome.storage.local.get("postHistory");
    const entry = postHistory.find(e => e.meta?.urn === result.urn);
    if (entry) {
      capturedMeta = entry.meta || { ...EMPTY_META };
      capturedPostText = entry.postText || "";
      renderResults(entry.data);
    }
  }
}

// ── Activation ────────────────────────────────────────────────────────────────

function isValidEmail(value) {
  return /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(value.trim());
}

activateBtn.addEventListener("click", async () => {
  const email = activationEmail.value.trim();
  if (!isValidEmail(email)) {
    activationEmail.classList.add("input-error");
    activationError.classList.remove("hidden");
    activationEmail.focus();
    return;
  }
  activationEmail.classList.remove("input-error");
  activationError.classList.add("hidden");
  activateBtn.disabled = true;
  activateBtn.textContent = "Activating…";
  await chrome.storage.sync.set({ userEmail: email });
  registerActivationInSupabase(email); // fire-and-forget
  activated = true;
  showState(true, false, false, false, false);
  checkPendingPost();
  checkPendingHistory();
});

activationEmail.addEventListener("input", () => {
  if (activationEmail.classList.contains("input-error")) {
    activationEmail.classList.remove("input-error");
    activationError.classList.add("hidden");
  }
});

activationEmail.addEventListener("keydown", (e) => {
  if (e.key === "Enter") activateBtn.click();
});

async function checkActivation() {
  const { userEmail } = await chrome.storage.sync.get("userEmail");
  if (userEmail) {
    activated = true;
    showState(true, false, false, false, false);
    checkPendingPost();
    checkPendingHistory();
  } else {
    showActivationState();
  }
}

checkActivation();
